const express = require('express');
const cors = require('cors');
const helmet = require("helmet");
const rateLimit = require("express-rate-limit");
const mongoSanitize = require("express-mongo-sanitize");
const hpp = require("hpp");
const xss = require("xss");

const sanitizeInput = (req, res, next) => {
    const sanitize = (data) => (typeof data === "string" ? xss(data) : data);

    if (req.body) req.body = Object.fromEntries(Object.entries(req.body).map(([key, value]) => [key, sanitize(value)]));
    if (req.query) req.query = Object.fromEntries(Object.entries(req.query).map(([key, value]) => [key, sanitize(value)]));
    if (req.params) req.params = Object.fromEntries(Object.entries(req.params).map(([key, value]) => [key, sanitize(value)]));

    next();
};

const limiter = rateLimit({
    windowMs: 60 * 1000,
    max: 60,
    standardHeaders: true,
    legacyHeaders: false,
    message: { success: false, message: "Too many requests, please try again later." },
    handler: (req, res) => {
        res.status(429).json({ success: false, message: "Too many requests, please try again later." });
    },
});

const gallery = require("./routes/gallery.js");
const uploads = require("./routes/upload.js");

const app = express();
const PORT = 8083;

app.set("trust proxy", 1);

app.use(
    helmet({
        contentSecurityPolicy: false, // If you don't need CSP, disable it
        crossOriginEmbedderPolicy: false,
    })
);

app.use(cors());
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(limiter);
app.use(
    mongoSanitize({
        allowDots: true,
        replaceWith: "_",
    })
);
app.use(hpp());
app.use(sanitizeInput);

const connectDB = require('./config/database.js');

app.use("/api", uploads);
app.use("/api", gallery);

app.get('/', (req, res) => {
    res.send(`
        <html>
            <head>
                <title>Welcome</title>
            </head>
            <style>
                * {
                    position: relative;
                    margin: 0;
                    padding: 0;
                    box-sizing: border-box;
                    font-family: 'Lato', sans-serif;
                }

                body {
                    height: 100vh;
                    display: flex;
                    flex-direction: column;
                    justify-content: center;
                    align-items: center;
                    background: linear-gradient(to bottom right, #EEE, #AAA);
                }

                h1 {
                    margin: 40px 0 20px;
                }

                .lock {
                    border-radius: 5px;
                    width: 55px;
                    height: 45px;
                    background-color: #333;
                    animation: dip 1s;
                    animation-delay: 1.5s;
                    top: -30px;
                    border: 5px solid #333;
                    border-bottom-color: transparent;
                    border-radius: 15px 15px 0 0;
                    height: 40px;
                }

                .lock::before,
                .lock::after {
                    content: '';
                    position: absolute;
                    border-left: 5px solid #333;
                    height: 20px;
                    width: 15px;
                    left: calc(50% - 12.5px);
                }

                .lock::before {
                    top: -30px;
                    border: 5px solid #333;
                    border-bottom-color: transparent;
                    border-radius: 15px 15px 0 0;
                    height: 30px;
                    animation: lock 2s, spin 2s;
                }

                .lock::after {
                    top: -10px;
                    border-right: 5px solid transparent;
                    animation: spin 2s;
                }

                @keyframes lock {
                    0% {
                        top: -45px;
                    }
                    65% {
                        top: -45px;
                    }
                    100% {
                        top: -30px;
                    }
                }

                @keyframes spin {
                    0% {
                        transform: scaleX(-1);
                        left: calc(50% - 30px);
                    }
                    65% {
                        transform: scaleX(1);
                        left: calc(50% - 12.5px);
                    }
                }

                @keyframes dip {
                    0% {
                        transform: translateY(0px);
                    }
                    50% {
                        transform: translateY(10px);
                    }
                    100% {
                        transform: translateY(0px);
                    }
                }

            </style>
            <body>
                <div class="lock"></div>
                <div class="message">
                <h1>Access to this page is restricted</h1>
                <p>Please check with the site admin if you believe this is a mistake.</p>
                </div>
            </body>
        </html>
    `);
});

app.listen(PORT, () => {
    connectDB();
    console.log(`Server listening on port ${PORT}`);
});