const axios = require('axios');
const FormData = require('form-data');
const { Readable } = require('stream');
const Gallery = require('../models/Gallery');
require('dotenv').config();

const createSlug = (name) => {
    if (!name || typeof name !== "string") return null;

    return name
        .toLowerCase()
        .replace(/\s+/g, '-')      // Replace spaces with hyphens
        .replace(/[^\w\-]+/g, '')  // Remove special characters
        .replace(/^-+|-+$/g, '');  // Remove leading/trailing hyphens
};

const bufferToStream = (buffer) => {
    const readable = new Readable();
    readable.push(buffer);
    readable.push(null);
    return readable;
};

const cloudflareUpload = async (req, res, next) => {
    console.log(req)
    if (!req.files || req.files.length === 0) {
        return res.status(400).json({ success: false, message: "Images are required" });
    }

    if (!req.body.name) {
        return res.status(400).json({ success: false, message: "Gallery name is required" });
    }

    try {
        const name = createSlug(req.body.name);
        const uploadedImages = [];

        await Promise.all(req.files.map(async (file, index) => {
            try {
                let formData = new FormData();
                const uniqueId = `${name}_gallery_${index + 1}_${Date.now()}`; // Ensure uniqueness
                
                const fileStream = bufferToStream(file.buffer);

                formData.append('file', fileStream, file.originalname);
                formData.append('id', uniqueId);

                const { data: response } = await axios.post(process.env.CLOUDFLARE_API_URL, formData, {
                    headers: {
                        'Authorization': `Bearer ${process.env.CLOUDFLARE_API_TOKEN}`,
                        ...formData.getHeaders(),
                    },
                });

                uploadedImages.push(response.result.id);

            } catch (uploadError) {
                console.error(`Failed to upload image ${file.originalname}:`, uploadError.message);
            }
        }));

        if (uploadedImages.length === 0) {
            return res.status(500).json({ success: false, message: "Failed to upload images to Cloudflare" });
        }

        req.body.images = uploadedImages;
        next();
    } catch (error) {
        console.error("Cloudflare Upload Error:", error.message);
        return res.status(500).json({ success: false, message: "Failed to upload images to Cloudflare" });
    }
};

const cloudflareDelete = async (req, res, next) => {

    try {
        const { name } = req.params; 

        if (!name) {
            return res.status(400).json({ success: false, message: "Image name is required" });
        }

        const galleries = await Gallery.find({ name });

        if (!galleries.length) {
            return res.status(404).json({ success: false, message: "No images found for this name" });
        }

        const imageIds = galleries.map((gallery) => gallery.image);

        const failedDeletions = [];

        await Promise.all(imageIds.map(async (imageId) => {
            try {
                await axios.delete(`${process.env.CLOUDFLARE_API_URL}/${imageId}`, {
                    headers: {
                        'Authorization': `Bearer ${process.env.CLOUDFLARE_API_TOKEN}`,
                    },
                });
            } catch (error) {
                console.error(`Failed to delete image ${imageId} from Cloudflare:`, error.message);
            }
        }));
        
        if (failedDeletions.length > 0) {
            return res.status(500).json({
                success: false,
                message: "Some images failed to delete from Cloudflare",
                failedImages: failedDeletions
            });
        }

        next();

    } catch (error) {
        console.error("Cloudflare Delete Error:", error.message);
        return res.status(500).json({ success: false, message: "Failed to delete images to Cloudflare" });
    }
}


module.exports = {
    cloudflareUpload,
    cloudflareDelete,
}